<?php
/**
 * Image Compressor for Image Compression Plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICP_Compressor {
    
    private $tinypng_api;
    
    public function __construct() {
        $this->tinypng_api = new ICP_TinyPNG_API();
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // Hook into image generation for different sizes - this runs after upload is complete
        add_filter('wp_generate_attachment_metadata', array($this, 'compress_generated_images'), 10, 2);
        
        // Add AJAX handlers for bulk compression
        add_action('wp_ajax_icp_bulk_compress', array($this, 'ajax_bulk_compress'));
        add_action('wp_ajax_icp_compress_single', array($this, 'ajax_compress_single'));
        add_action('wp_ajax_icp_get_recent_compressions', array($this, 'ajax_get_recent_compressions'));

        // Add AJAX handlers for restore functionality
        add_action('wp_ajax_icp_restore_single', array($this, 'ajax_restore_single'));
        add_action('wp_ajax_icp_bulk_restore', array($this, 'ajax_bulk_restore'));
        
        // Hook into attachment deletion to clean up compression records
        add_action('delete_attachment', array($this, 'cleanup_compression_data'));
    }
    

    
    public function compress_generated_images($metadata, $attachment_id) {
        // Check if auto-compression is enabled
        $settings = get_option('icp_settings');
        if (!isset($settings['auto_compress']) || !$settings['auto_compress']) {
            return $metadata;
        }
        
        // Check if API key is configured
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';
        
        // Check if this image has already been compressed
        $existing_compression = ICP_Database::get_compression_stats($attachment_id);
        if ($existing_compression) {
            return $metadata; // Already compressed, skip
        }
        
        $base_file = get_attached_file($attachment_id);
        
        if (!$base_file || !$this->is_supported_image_file($base_file)) {
            return $metadata;
        }
        
        // Set a flag to prevent infinite loops
        if (get_transient('icp_compressing_' . $attachment_id)) {
            return $metadata;
        }
        set_transient('icp_compressing_' . $attachment_id, true, 300); // 5 minutes
        
        // If API key is missing, store info for notification popup but don't compress
        if (empty($api_key)) {
            // Store this upload for popup notification about missing API key
            $this->store_recent_upload_for_popup($attachment_id, 'missing_api_key');
            
            // Clean up the transient
            delete_transient('icp_compressing_' . $attachment_id);
            
            return $metadata;
        }
        
        // Only compress the main uploaded image, not the generated thumbnails
        // This prevents multiple API calls for a single image upload
        $result = $this->compress_image_file($base_file, $attachment_id);
        
        // Clean up the transient
        delete_transient('icp_compressing_' . $attachment_id);
        
        // Update the metadata with the new file size if compression was successful
        if ($result && $result['success'] && isset($metadata['filesize'])) {
            $metadata['filesize'] = $result['compressed_size'];
            
            // Store this compression for popup notification
            $this->store_recent_compression_for_popup($attachment_id);
        } elseif ($result && !$result['success']) {
            // Store compression failure for popup notification
            $this->store_recent_upload_for_popup($attachment_id, 'compression_failed', $result['error']);
        }
        
        return $metadata;
    }
    
    public function compress_image_file($file_path, $attachment_id = null) {
        if (!file_exists($file_path)) {
            return false;
        }

        // Get the original file size before compression
        $original_size = filesize($file_path);

        // Backup original image before compression (if attachment ID is provided)
        $backup_path = null;
        if ($attachment_id) {
            $backup_result = ICP_Backup::backup_original($attachment_id, $file_path);
            if ($backup_result['success']) {
                $backup_path = $backup_result['backup_path'];
            } else {
                // Log backup failure but continue with compression
                error_log('ICP: Failed to backup original image for attachment ' . $attachment_id . ': ' . $backup_result['error']);
            }
        }

        $result = $this->tinypng_api->compress_image($file_path);

        // Save compression data to database if we have an attachment ID
        if ($attachment_id && $result !== false) {
            $compression_data = array(
                'attachment_id' => $attachment_id,
                'original_size' => $original_size, // Use the size we measured before compression
                'compressed_size' => isset($result['compressed_size']) ? $result['compressed_size'] : $original_size,
                'savings_percent' => 0,
                'status' => $result['success'] ? 'success' : 'failed',
                'backup_path' => $backup_path
            );

            // Calculate savings percent
            if ($result['success'] && $original_size > 0) {
                $compression_data['savings_percent'] = (($original_size - $compression_data['compressed_size']) / $original_size) * 100;
            }

            if (!$result['success']) {
                $compression_data['error_message'] = $result['error'];

                // If compression failed and we created a backup, clean it up
                if ($backup_path && file_exists($backup_path)) {
                    ICP_Backup::delete_backup($backup_path);
                }
            }

            ICP_Database::save_compression_data($compression_data);

            // Update WordPress attachment metadata with new file size if compression was successful
            if ($result['success'] && isset($result['compressed_size'])) {
                $this->update_attachment_metadata($attachment_id, $result['compressed_size']);
            }
        }

        return $result;
    }
    
    public function ajax_bulk_compress() {
        check_ajax_referer('icp_bulk_compress', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        // Check if API key is configured before starting bulk compression
        $settings = get_option('icp_settings');
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(array(
                'message' => 'TinyPNG API key is required for bulk compression.',
                'action_required' => 'configure_api_key',
                'settings_url' => admin_url('admin.php?page=image-compression-settings'),
                'error_type' => 'missing_api_key'
            ));
            return;
        }

        $batch_size = 5; // Process 5 images at a time
        $offset = intval($_POST['offset']);
        $failed_only = isset($_POST['failed_only']) && $_POST['failed_only'] === 'true';

        // Get bulk compression settings for folder exclusion
        $bulk_settings = get_option('icp_bulk_settings', array());
        $excluded_folders = isset($bulk_settings['excluded_folders']) ? $bulk_settings['excluded_folders'] : array();

        // Get images based on whether we're retrying failed only or all uncompressed
        // Get images list only on the first batch (offset 0)
        // For subsequent batches, use the total from the first call
        if ($offset === 0) {
            if ($failed_only) {
                $uncompressed_images = ICP_Database::get_failed_compression_images($excluded_folders);
                $transient_suffix = '_failed';
            } else {
                $uncompressed_images = ICP_Database::get_uncompressed_images($excluded_folders);
                $transient_suffix = '_all';
            }
            $total_images = count($uncompressed_images);
            
            // Store the initial list in a transient for subsequent batches
            set_transient('icp_bulk_images' . $transient_suffix . '_' . get_current_user_id(), $uncompressed_images, HOUR_IN_SECONDS);
            set_transient('icp_bulk_total' . $transient_suffix . '_' . get_current_user_id(), $total_images, HOUR_IN_SECONDS);
        } else {
            // Get the original list from transient
            $transient_suffix = $failed_only ? '_failed' : '_all';
            $uncompressed_images = get_transient('icp_bulk_images' . $transient_suffix . '_' . get_current_user_id());
            $total_images = get_transient('icp_bulk_total' . $transient_suffix . '_' . get_current_user_id());
            
            if (!$uncompressed_images || !$total_images) {
                // Fallback if transient expired - get fresh list
                if ($failed_only) {
                    $uncompressed_images = ICP_Database::get_failed_compression_images($excluded_folders);
                } else {
                    $uncompressed_images = ICP_Database::get_uncompressed_images($excluded_folders);
                }
                $total_images = count($uncompressed_images) + $offset; // Estimate original total
            }
        }
        
        // Debug logging
        error_log("ICP Bulk Compress: Processing batch at offset $offset, total images: $total_images, available images: " . count($uncompressed_images));
        
        // Get the batch to process
        $batch_images = array_slice($uncompressed_images, $offset, $batch_size);
        $processed = 0;
        $successful_in_batch = 0;
        $errors = array();
        
        error_log("ICP Bulk Compress: Batch size: " . count($batch_images));
        
        // Process each image in the batch
        foreach ($batch_images as $image) {
            // Check if this image was already successfully compressed (skip it)
            $existing_compression = ICP_Database::get_compression_stats($image->ID);
            if ($existing_compression && $existing_compression->status === 'success') {
                error_log("ICP Bulk Compress: Image ID {$image->ID} already successfully compressed, skipping");
                $processed++;
                continue;
            }
            
            // If the image had a previous failed attempt, log that we're retrying
            if ($existing_compression && $existing_compression->status === 'failed') {
                error_log("ICP Bulk Compress: Retrying previously failed image ID {$image->ID}");
            }
            
            $file_path = get_attached_file($image->ID);
            
            if ($file_path && file_exists($file_path) && $this->is_supported_image_file($file_path)) {
                $result = $this->compress_image_file($file_path, $image->ID);
                
                if ($result && $result['success']) {
                    $successful_in_batch++;
                    error_log("ICP Bulk Compress: Successfully compressed image ID {$image->ID}");
                } else {
                    $error_msg = $result ? $result['error'] : 'Unknown compression error';
                    error_log("ICP Bulk Compress: Failed to compress image ID {$image->ID}: $error_msg");
                    $errors[] = array(
                        'id' => $image->ID,
                        'title' => $image->post_title,
                        'error' => $error_msg
                    );
                }
            } else {
                // File doesn't exist or is not supported
                $error_msg = 'File not found or unsupported format';
                error_log("ICP Bulk Compress: Skipped image ID {$image->ID}: $error_msg");
                $errors[] = array(
                    'id' => $image->ID,
                    'title' => $image->post_title,
                    'error' => $error_msg
                );
            }
            
            $processed++;
        }
        
        // Calculate new offset
        $new_offset = $offset + $batch_size;
        $is_completed = $new_offset >= $total_images;
        
        // Clean up transients if completed
        if ($is_completed) {
            $transient_suffix = $failed_only ? '_failed' : '_all';
            delete_transient('icp_bulk_images' . $transient_suffix . '_' . get_current_user_id());
            delete_transient('icp_bulk_total' . $transient_suffix . '_' . get_current_user_id());
        }
        
        error_log("ICP Bulk Compress: New offset: $new_offset, Total: $total_images, Completed: " . ($is_completed ? 'true' : 'false'));
        
        // Get current statistics from database for more accurate reporting
        $total_stats = ICP_Database::get_total_savings();
        $total_errors_in_db = ICP_Database::get_total_compression_errors();
        
        $response = array(
            'success' => true,
            'processed' => $processed,
            'successful_in_batch' => $successful_in_batch,
            'total' => $total_images,
            'offset' => $new_offset,
            'completed' => $is_completed,
            'errors' => $errors,
            'batch_errors' => count($errors),
            'total_compressed_so_far' => $total_stats ? $total_stats->total_images : 0,
            'total_errors_so_far' => $total_errors_in_db
        );
        
        error_log("ICP Bulk Compress: Response - " . json_encode($response));
        
        wp_send_json($response);
    }
    
    public function ajax_compress_single() {
        check_ajax_referer('icp_compress_single', 'nonce');
        
        if (!current_user_can('upload_files')) {
            wp_die('Insufficient permissions');
        }
        
        // Check if API key is configured before attempting compression
        $settings = get_option('icp_settings');
        $api_key = isset($settings['api_key']) ? trim($settings['api_key']) : '';
        
        if (empty($api_key)) {
            wp_send_json_error(array(
                'message' => 'TinyPNG API key is required to compress images.',
                'action_required' => 'configure_api_key',
                'settings_url' => admin_url('admin.php?page=image-compression-settings'),
                'error_type' => 'missing_api_key'
            ));
            return;
        }

        $attachment_id = intval($_POST['attachment_id']);
        $file_path = get_attached_file($attachment_id);
        
        if (!$file_path || !file_exists($file_path)) {
            wp_send_json_error('Image file not found');
        }
        
        $result = $this->compress_image_file($file_path, $attachment_id);
        
        if ($result && $result['success']) {
            $compression_data = ICP_Database::get_compression_stats($attachment_id);
            wp_send_json_success(array(
                'message' => 'Image compressed successfully',
                'original_size' => $this->format_bytes($compression_data->original_size),
                'compressed_size' => $this->format_bytes($compression_data->compressed_size),
                'savings_percent' => round($compression_data->savings_percent, 1),
                'savings_bytes' => $this->format_bytes($compression_data->original_size - $compression_data->compressed_size)
            ));
        } else {
            wp_send_json_error($result ? $result['error'] : 'Unknown compression error');
        }
    }
    
    private function is_supported_image($mime_type) {
        $supported_types = array(
            'image/jpeg',
            'image/png',
            'image/gif',
            'image/webp'
        );
        
        return in_array($mime_type, $supported_types);
    }
    
    private function is_supported_image_file($file_path) {
        $mime_type = wp_check_filetype($file_path)['type'];
        return $this->is_supported_image($mime_type);
    }
    
    private function format_bytes($bytes) {
        if ($bytes >= 1048576) {
            return round($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return round($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }
    
    /**
     * Update WordPress attachment metadata with new file size after compression
     */
    private function update_attachment_metadata($attachment_id, $new_file_size) {
        // Get current attachment metadata
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        if (!$metadata) {
            $metadata = array();
        }
        
        // Update the filesize in metadata
        $metadata['filesize'] = $new_file_size;
        
        // Save the updated metadata back to WordPress
        wp_update_attachment_metadata($attachment_id, $metadata);
        
        // Also update the post meta directly for compatibility
        update_post_meta($attachment_id, '_wp_attachment_metadata', $metadata);
        
        // Log the update for debugging
        error_log("ICP: Updated attachment metadata for ID {$attachment_id} with new file size: {$new_file_size} bytes");
    }
    
    /**
     * Store recent compression data for popup notifications
     */
    private function store_recent_compression_for_popup($attachment_id) {
        $user_id = get_current_user_id();
        if (!$user_id) return;
        
        // Get existing recent compressions for this user
        $recent_compressions = get_transient('icp_recent_compressions_' . $user_id);
        if (!$recent_compressions) {
            $recent_compressions = array();
        }
        
        // Add this compression to the list
        $recent_compressions[] = array(
            'attachment_id' => $attachment_id,
            'timestamp' => time(),
            'type' => 'compression_success'
        );
        
        // Keep only the last 5 compressions and only those from the last 10 minutes
        $recent_compressions = array_filter($recent_compressions, function($item) {
            return (time() - $item['timestamp']) < 600; // 10 minutes
        });
        $recent_compressions = array_slice($recent_compressions, -5); // Keep last 5
        
        // Store for 10 minutes
        set_transient('icp_recent_compressions_' . $user_id, $recent_compressions, 600);
    }
    
    /**
     * Store recent upload data for popup notifications when API key is missing or compression fails
     */
    private function store_recent_upload_for_popup($attachment_id, $notification_type = 'missing_api_key', $error_message = null) {
        $user_id = get_current_user_id();
        if (!$user_id) return;
        
        // Get existing recent compressions for this user
        $recent_compressions = get_transient('icp_recent_compressions_' . $user_id);
        if (!$recent_compressions) {
            $recent_compressions = array();
        }
        
        // Add this upload to the list
        $upload_data = array(
            'attachment_id' => $attachment_id,
            'timestamp' => time(),
            'type' => $notification_type
        );
        
        // Add error message for compression failures
        if ($notification_type === 'compression_failed' && $error_message) {
            $upload_data['error_message'] = $error_message;
        }
        
        $recent_compressions[] = $upload_data;
        
        // Keep only the last 5 items and only those from the last 10 minutes
        $recent_compressions = array_filter($recent_compressions, function($item) {
            return (time() - $item['timestamp']) < 600; // 10 minutes
        });
        $recent_compressions = array_slice($recent_compressions, -5); // Keep last 5
        
        // Store for 10 minutes
        set_transient('icp_recent_compressions_' . $user_id, $recent_compressions, 600);
    }
    
    /**
     * AJAX handler to get recent compression data for popup notifications
     */
    public function ajax_get_recent_compressions() {
        check_ajax_referer('icp_get_recent_compressions', 'nonce');
        
        if (!current_user_can('upload_files')) {
            wp_die('Insufficient permissions');
        }
        
        $user_id = get_current_user_id();
        $recent_compressions = get_transient('icp_recent_compressions_' . $user_id);
        
        if (!$recent_compressions) {
            wp_send_json_success(array('compressions' => array()));
            return;
        }
        
        $compression_details = array();
        
        foreach ($recent_compressions as $item) {
            $attachment = get_post($item['attachment_id']);
            $title = $attachment ? $attachment->post_title : 'Unknown Image';
            
            if ($item['type'] === 'compression_success') {
                $compression_data = ICP_Database::get_compression_stats($item['attachment_id']);
                
                if ($compression_data && $compression_data->status === 'success') {
                    $compression_details[] = array(
                        'attachment_id' => $item['attachment_id'],
                        'title' => $title,
                        'original_size' => $this->format_bytes($compression_data->original_size),
                        'compressed_size' => $this->format_bytes($compression_data->compressed_size),
                        'savings_bytes' => $this->format_bytes($compression_data->original_size - $compression_data->compressed_size),
                        'savings_percent' => round($compression_data->savings_percent, 1),
                        'timestamp' => $item['timestamp'],
                        'type' => 'compression_success'
                    );
                }
            } elseif ($item['type'] === 'missing_api_key') {
                // Get file size for uploaded image
                $file_path = get_attached_file($item['attachment_id']);
                $file_size = file_exists($file_path) ? filesize($file_path) : 0;
                
                $compression_details[] = array(
                    'attachment_id' => $item['attachment_id'],
                    'title' => $title,
                    'file_size' => $this->format_bytes($file_size),
                    'timestamp' => $item['timestamp'],
                    'type' => 'missing_api_key',
                    'settings_url' => admin_url('admin.php?page=image-compression-settings')
                );
            } elseif ($item['type'] === 'compression_failed') {
                // Get file size for uploaded image
                $file_path = get_attached_file($item['attachment_id']);
                $file_size = file_exists($file_path) ? filesize($file_path) : 0;
                
                $compression_details[] = array(
                    'attachment_id' => $item['attachment_id'],
                    'title' => $title,
                    'file_size' => $this->format_bytes($file_size),
                    'timestamp' => $item['timestamp'],
                    'type' => 'compression_failed',
                    'error_message' => isset($item['error_message']) ? $item['error_message'] : 'Unknown compression error'
                );
            }
        }
        
        // Clear the recent compressions after retrieving them
        delete_transient('icp_recent_compressions_' . $user_id);
        
        wp_send_json_success(array('compressions' => $compression_details));
    }
    
    /**
     * Hook into attachment deletion to clean up compression records
     */
    public function cleanup_compression_data($attachment_id) {
        // Get backup path before deleting database record
        $compression_stats = ICP_Database::get_compression_stats($attachment_id);
        if ($compression_stats && !empty($compression_stats->backup_path)) {
            ICP_Backup::delete_backup($compression_stats->backup_path);
        }

        // Clean up compression data from database
        ICP_Database::delete_compression_data($attachment_id);

        // Clean up compression data from transients
        delete_transient('icp_compressing_' . $attachment_id);
        delete_transient('icp_recent_compressions_' . get_current_user_id());
    }

    /**
     * Restore original image from backup
     */
    public function restore_image($attachment_id) {
        // Get compression stats to find backup path
        $compression_stats = ICP_Database::get_compression_stats($attachment_id);

        if (!$compression_stats) {
            return array(
                'success' => false,
                'error' => 'No compression data found for this image'
            );
        }

        if (empty($compression_stats->backup_path)) {
            return array(
                'success' => false,
                'error' => 'No backup file available for this image'
            );
        }

        if (!ICP_Backup::backup_exists($compression_stats->backup_path)) {
            return array(
                'success' => false,
                'error' => 'Backup file not found or not accessible'
            );
        }

        // Get current file path
        $current_file_path = get_attached_file($attachment_id);
        if (!$current_file_path) {
            return array(
                'success' => false,
                'error' => 'Current image file not found'
            );
        }

        // Restore the original image
        $restore_result = ICP_Backup::restore_original(
            $attachment_id,
            $compression_stats->backup_path,
            $current_file_path
        );

        if ($restore_result['success']) {
            // Update database to mark as restored
            ICP_Database::update_restore_status($attachment_id, true);

            // Get new file size after restore
            $restored_size = filesize($current_file_path);

            return array(
                'success' => true,
                'message' => 'Image restored successfully',
                'original_size' => $compression_stats->original_size,
                'restored_size' => $restored_size
            );
        } else {
            return $restore_result;
        }
    }

    /**
     * AJAX handler for single image restore
     */
    public function ajax_restore_single() {
        check_ajax_referer('icp_restore_single', 'nonce');

        if (!current_user_can('upload_files')) {
            wp_die('Insufficient permissions');
        }

        $attachment_id = intval($_POST['attachment_id']);

        if (!$attachment_id) {
            wp_send_json_error('Invalid attachment ID');
            return;
        }

        $result = $this->restore_image($attachment_id);

        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }

    /**
     * AJAX handler for bulk restore
     */
    public function ajax_bulk_restore() {
        check_ajax_referer('icp_bulk_restore', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }

        $attachment_ids = isset($_POST['attachment_ids']) ? array_map('intval', $_POST['attachment_ids']) : array();

        if (empty($attachment_ids)) {
            wp_send_json_error('No images selected for restore');
            return;
        }

        $results = array(
            'success' => 0,
            'failed' => 0,
            'errors' => array()
        );

        foreach ($attachment_ids as $attachment_id) {
            $result = $this->restore_image($attachment_id);

            if ($result['success']) {
                $results['success']++;
            } else {
                $results['failed']++;
                $results['errors'][] = array(
                    'attachment_id' => $attachment_id,
                    'error' => $result['error']
                );
            }
        }

        wp_send_json_success($results);
    }
}